#***********************************************************************
# JTE_DOM
#	DOM for JTE
#***********************************************************************

use strict ;
use warnings ;


package JTE_DOM ;


our $XML_ELEMENT_NODE        = 1 ;
our $XML_TEXT_NODE           = 3 ;
our $XML_CDATA_SECTION_NODE  = 4 ;
our $XML_COMMENT_NODE        = 8 ;
our $XML_DOCUMENT_NODE       = 9 ;


#=======================================================================
# JTE_DOMList

package JTE_DOMList ;


sub new {
	bless {
		'_item' => [],
		'_index' => {}
	},
	$_[ 0 ] ;
}


sub length {
	my $this = shift ;
	
	return ( my $n = @{ $this->{ '_item' } } ) ;
}


sub item {
	my $this = shift ;
	my $n = shift ;
	
	if ( @_ ) {
		$this->{ '_item' }->[ $n ] = $_[ 0 ] ;
		return ;
	}
	
	return $this->{ '_item' }->[ $n ] ;
}


#=======================================================================
# JTE_DOMAttributeList

package JTE_DOMAttributeList ;
@JTE_DOMAttributeList::ISA = qw(JTE_DOMList) ;


#=======================================================================
# JTE_DOMNodeList

package JTE_DOMNodeList ;
@JTE_DOMNodeList::ISA = qw(JTE_DOMList) ;


#=======================================================================
# JTE_DOMAttribute

package JTE_DOMAttribute ;


sub new {
	my $class = shift ;
	my $name = shift ;
	my $value = shift ;
	
	bless {
		'name' => $name,
		'value' => $value
	},
	$class ;
}


sub name {
	$_[ 0 ]->{ 'name' } ;
}


sub value {
	$_[ 0 ]->{ 'value' } ;
}


#=======================================================================
# JTE_DOMNode

package JTE_DOMNode ;


our $seq = 0 ;


sub new {
	my $class = shift ;
	my $nodeType = shift ;
	
	$seq++ ;
	
	bless {
		'seq'		=> $seq,
		'nodeType'	=> $nodeType
	},
	$class ;
}


sub nodeType {
	$_[ 0 ]->{ 'nodeType' } ;
}


sub ownerDocument {
	$_[ 0 ]->{ 'ownerDocument' } ;
}


sub parentNode {
	$_[ 0 ]->{ 'parentNode' } ;
}


#=======================================================================
# JTE_DOMText

package JTE_DOMText ;
@JTE_DOMText::ISA = qw(JTE_DOMNode) ;


sub new {
	my $class = shift ;
	my $text = shift ;
	
	my $this = $class->SUPER::new( $JTE_DOM::XML_TEXT_NODE ) ;
	
	$this->{ 'nodeValue' } = $text ;
	
	bless $this, $class ;
}


sub cloneNode {
	my $this = shift ;
	
	my $node = new JTE_DOMText( $this->{ 'nodeValue' } ) ;
	$node->{ 'ownerDocument' } = $this->{ 'ownerDocument' } ;
	
	return $node ;
}


sub nodeValue:lvalue {
	$_[ 0 ]->{ 'nodeValue' } ;
}


#=======================================================================
# JTE_DOMComment

package JTE_DOMComment ;
@JTE_DOMComment::ISA = qw(JTE_DOMNode) ;


sub new {
	my $class = shift ;
	my $comment = shift ;
	
	my $this = $class->SUPER::new( $JTE_DOM::XML_COMMENT_NODE ) ;
	
	$this->{ 'nodeValue' } = $comment ;
	
	bless $this, $class ;
}


sub cloneNode {
	my $this = shift ;
	
	my $node = new JTE_DOMComment( $this->{ 'nodeValue' } ) ;
	$node->{ 'ownerDocument' } = $this->{ 'ownerDocument' } ;
	
	return $node ;
}


sub nodeValue:lvalue {
	$_[ 0 ]->{ 'nodeValue' } ;
}


#=======================================================================
# JTE_DOMCDATASection

package JTE_DOMCDATASection ;
@JTE_DOMCDATASection::ISA = qw(JTE_DOMNode) ;


sub new {
	my $class = shift ;
	my $cdata = shift ;
	
	my $this = $class->SUPER::new( $JTE_DOM::XML_CDATA_SECTION_NODE ) ;
	
	$this->{ 'nodeValue' } = $cdata ;
	
	bless $this, $class ;
}


sub cloneNode {
	my $this = shift ;
	
	my $node = new JTE_DOMCDATASection( $this->{ 'nodeValue' } ) ;
	$node->{ 'ownerDocument' } = $this->{ 'ownerDocument' } ;
	
	return $node ;
}


sub nodeValue:lvalue {
	$_[ 0 ]->{ 'nodeValue' } ;
}


#=======================================================================
# JTE_DOMElement

package JTE_DOMElement ;
@JTE_DOMElement::ISA = qw(JTE_DOMNode) ;


sub new {
	my $class = shift ;
	my $tagName = shift ;
	
	my $this = $class->SUPER::new( $JTE_DOM::XML_ELEMENT_NODE ) ;
	
	$this->{ 'tagName' } = $tagName ;
	$this->{ 'attributes' } = new JTE_DOMAttributeList() ;
	$this->{ 'childNodes' } = new JTE_DOMNodeList() ;
	
	bless $this, $class ;
}


sub tagName {
	$_[ 0 ]->{ 'tagName' } ;
}


sub attributes {
	$_[ 0 ]->{ 'attributes' } ;
}


sub hasAttributes {
    return ( $_[ 0 ]->attributes->length > 0 ) ;
}


sub childNodes {
	$_[ 0 ]->{ 'childNodes' } ;
}


sub cloneNode {
	my $this = shift ;
	my $deep = shift ;
	
	my $node = new JTE_DOMElement( $this->tagName ) ;
	$node->{ 'ownerDocument' } = $this->{ 'ownerDocument' } ;
	
	for ( my $n = 0 ; $n < $this->attributes->length ; $n++ ) {
		my $attribute = $this->attributes->item( $n ) ;
		$node->setAttribute( $attribute->name, $attribute->value ) ;
	}
	
	if ( $deep ) {
		for ( my $n = 0 ; $n < $this->childNodes->length ; $n++ ) {
			$node->appendChild( $this->childNodes->item( $n )->cloneNode( 1 ) ) ;
		}
	}
	
	return $node ;
}


sub hasAttribute {
	my $this = shift ;
	my $name = shift ;
	
    return ( defined( $this->attributes->{ '_index' }->{ $name } ) ) ;
}


sub setAttribute {
	my $this = shift ;
	my $name = shift ;
	my $value = shift ;
	
    if ( $this->hasAttribute( $name ) ) {
    	$this->attributes->{ '_item' }->[ $this->attributes->{ '_index' }->{ $name } ]->{ 'value' } = $value ;
    } else {
    	$this->attributes->{ '_index' }->{ $name } = @{ $this->attributes->{ '_item' } } ;
    	push( @{ $this->attributes->{ '_item' } }, new JTE_DOMAttribute( $name, $value ) ) ;
    }
}


sub getAttribute {
	my $this = shift ;
	my $name = shift ;
	
    if ( $this->hasAttribute( $name ) ) {
		return $this->attributes->{ '_item' }->[ $this->attributes->{ '_index' }->{ $name } ]->value ;
    } else {
    	return '' ;
    }
}


sub removeAttribute {
	my $this = shift ;
	my $name = shift ;
	
    if ( $this->hasAttribute( $name ) ) {
      splice( @{ $this->attributes->{ '_item' } }, $this->attributes->{ '_index' }->{ $name }, 1 ) ;
      
      $this->attributes->{ '_index' } = {} ;
      
      for ( my $n = 0 ; $n < $this->attributes->length ; $n++ ) {
	      $this->attributes->{ '_index' }->{ $this->attributes->item( $n )->name } = $n ;
      }
    }
}


sub appendChild {
	my $this = shift ;
	my $node = shift ;
	
	$node->{ 'parentNode' } = $this ;
	
	$this->childNodes->{ '_index' }->{ $node->{ 'seq' } } = $this->childNodes->length ;
	push( @{ $this->childNodes->{ '_item' } }, $node ) ;
}


sub insertBefore {
	my $this = shift ;
	my $node = shift ;
	my $before = shift ;
	
	if ( $before->parentNode->{ 'seq' } != $this->{ 'seq' } ) {
		print "insertBefor: ", $this->{ 'seq' }, " != ", $before->parentNode->{ 'seq' }, ", ", $before->{ 'seq' }, "<br />\n" ;
		return ;
	}
	
	if ( defined( $this->childNodes->{ '_index' }->{ $before->{ 'seq' } } ) ) {
		my $n = $this->childNodes->{ '_index' }->{ $before->{ 'seq' } } ;
		
		$node->{ 'parentNode' } = $this ;
		splice( @{ $this->childNodes->{ '_item' } }, $n, 0, ( $node ) ) ;
		
    	$this->childNodes->{ '_index' } = {} ;
    	
    	for ( my $n = 0 ; $n < $this->childNodes->length ; $n++ ) {
	    	$this->childNodes->{ '_index' }->{ $this->childNodes->item( $n )->{ 'seq' } } = $n ;
    	}
	}
}


sub replaceChild {
	my $this = shift ;
	my $node = shift ;
	my $replace = shift ;
	
	if ( $replace->parentNode->{ 'seq' } != $this->{ 'seq' } ) {
		return ;
	}
	
	if ( defined( $this->childNodes->{ '_index' }->{ $replace->{ 'seq' } } ) ) {
		my $n = $this->childNodes->{ '_index' }->{ $replace->{ 'seq' } } ;
		
		$node->{ 'parentNode' } = $this ;
		$this->childNodes->{ '_item' }->[ $n ] = $node ;
	}
}


sub removeChild {
	my $this = shift ;
	my $node = shift ;
	
	if ( $node->parentNode->{ 'seq' } != $this->{ 'seq' } ) {
		return ;
	}
	
	if ( defined( $this->childNodes->{ '_index' }->{ $node->{ 'seq' } } ) ) {
		my $n = $this->childNodes->{ '_index' }->{ $node->{ 'seq' } } ;
		
		splice( @{ $this->childNodes->{ '_item' } }, $n, 1 ) ;
		
    	$this->childNodes->{ '_index' } = {} ;
    	
    	for ( my $n = 0 ; $n < $this->childNodes->length ; $n++ ) {
	    	$this->childNodes->{ '_index' }->{ $this->childNodes->item( $n )->{ 'seq' } } = $n ;
    	}
	}
}


sub firstChild {
	my $this = shift ;
	
	if ( $this->childNodes->length == 0 ) {
		return ;
	}
	
	return $this->childNodes->item( 0 ) ;
}


sub lastChild {
	my $this = shift ;
	
	if ( $this->childNodes->length == 0 ) {
		return ;
	}
	
	return $this->childNodes->item( $this->childNodes->length - 1 ) ;
}


sub previousSibling {
	my $this = shift ;
	
	unless ( defined( $this->{ 'parentNode' } ) ) {
		return ;
	}
	
	unless ( $this->parentNode->childNodes->{ '_index' }->{ $this->{ 'seq' } } ) {
		return ;
	}
	
	my $n = $this->parentNode->childNodes->{ '_index' }->{ $this->{ 'seq' } } ;
	
	if ( $n == 0 ) {
		return ;
	}
	
	return $this->parentNode->childNodes->item( $n - 1 ) ;
}


sub nextSibling {
	my $this = shift ;
	
	unless ( defined( $this->{ 'parentNode' } ) ) {
		return ;
	}
	
	unless ( $this->parentNode->childNodes->{ '_index' }->{ $this->{ 'seq' } } ) {
		return ;
	}
	
	my $n = $this->parentNode->childNodes->{ '_index' }->{ $this->{ 'seq' } } ;
	
	if ( $n == 0 ) {
		return ;
	}
	
	if ( $n == ( $this->parentNode->childNodes->length - 1 ) ) {
		return ;
	}
	
	return $this->parentNode->childNodes->item( $n + 1 ) ;
}


#=======================================================================
# JTE_DOMDocumentType

package JTE_DOMDocumentType ;


sub new {
	my $class = shift ;
	my $docType = shift || '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">' ;
	my $this = {} ;
	
	my $obj = bless $this, $class ;
	
	$obj->parse( $docType ) ;
	
	$obj ;
}


sub parse {
	my $this = shift ;
	my $docType = shift || '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">' ;
	
	$this->{ 'docType' } = $docType ;
	
	if ( $this->{ 'docType' } =~ /\A\<!DOCTYPE[\s\n]+([^\s\n]+)[\s\n]+PUBLIC[\s\n]+("[^"]*")[\s\n]+("[^"]*")[\s\n]*\>\z/m ) {
		$this->{ 'internalId' } = $this->{ 'docType' } ;
		$this->{ 'name' } = lc( $1 ) ;
		$this->{ 'publicId' } = $2 ;
		$this->{ 'systemId' } = $3 ;
	}
}


#=======================================================================
# JTE_DOMDocument

package JTE_DOMDocument ;
@JTE_DOMDocument::ISA = qw(JTE_DOMNode) ;


sub new {
	my $class = shift ;
	
	my $this = $class->SUPER::new( $JTE_DOM::XML_DOCUMENT_NODE ) ;
	
	$this->{ 'docType' } = new JTE_DOMDocumentType() ;
	$this->{ 'encoding' } = 'UTF-8' ;
	
	bless $this, $class ;
}


sub documentElement {
	$_[ 0 ]->{ 'documentElement' } ;
}


sub createComment {
	my $this = shift ;
	my $comment = shift || '' ;
	
	my $node = new JTE_DOMComment( $comment ) ;
	$node->{ 'ownerDocument' } = $this ;
	
	return $node ;
}


sub createTextNode {
	my $this = shift ;
	my $text = shift || '' ;
	
	my $node = new JTE_DOMText( $text ) ;
	$node->{ 'ownerDocument' } = $this ;
	
	return $node ;
}


sub createCDATASection {
	my $this = shift ;
	my $cdata = shift || '' ;
	
	my $node = new JTE_DOMCDATASection( $cdata ) ;
	$node->{ 'ownerDocument' } = $this ;
	
	return $node ;
}


sub createElement {
	my $this = shift ;
	my $tagName = shift ;
	my $text = shift || undef ;
	
	my $node = new JTE_DOMElement( $tagName ) ;
	$node->{ 'ownerDocument' } = $this ;
	
	if ( defined( $text ) ) {
		$node->appendChild( $this->creteTextNode( $text ) ) ;
	}
	
	return $node ;
}


sub appendChild {
	my $this = shift ;
	my $node = shift ;
	
    $node->{ 'parentNode'} = $this ;
    $this->{ 'documentElement' } = $node ;
}


#=======================================================================
# saveXML

sub saveXML {
	my $this = shift ;
	
    my $xml = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" ;
    
    $this->_tree( \$xml, $this->{ 'documentElement' }, '' ) ;
    
    return $xml ;
}


sub _tree {
	my $this = shift ;
	my $xml_ref = shift ;
	my $node = shift ;
	my $tab = shift || '' ;
	
    if ( $node->nodeType == $JTE_DOM::XML_COMMENT_NODE ) {
        $$xml_ref .= "<!--" . $node->nodeValue . "-->" ;
        return ;
	}
	
    if ( $node->nodeType == $JTE_DOM::XML_CDATA_SECTION_NODE ) {
        $$xml_ref .= "<![CDATA[" . $node->nodeValue . "]]>" ;
        return ;
	}
	
    if ( $node->nodeType == $JTE_DOM::XML_TEXT_NODE ) {
    	my $text = $node->nodeValue ;
    	$text = $this->htmlspecialchars( $text ) ;
        $text =~ s/&#039;/&apos;/g ;
        $$xml_ref .= $text ;
        return ;
	}
	
    if ( $node->nodeType == $JTE_DOM::XML_ELEMENT_NODE ) {
        my @tag = ( $node->tagName ) ;
        
        for ( my $n = 0 ; $n < $node->attributes->length ; $n++ ) {
          my $name = $this->htmlspecialchars( $node->attributes->item( $n )->name ) ;
          my $value = $node->attributes->item( $n )->value ;
            
          push( @tag, $name . '="' . $value . '"' ) ;
        }
        
        if ( $node->childNodes->length == 0 ) {
            $$xml_ref .= "$tab<" . join( ' ', @tag ) . "/>\n" ;
        } else {
        	if ( ( $node->childNodes->length == 1 ) && ( $node->childNodes->item( 0 )->nodeType != $JTE_DOM::XML_ELEMENT_NODE ) ) {
	            $$xml_ref .= "$tab<" . join( ' ', @tag ) . ">" ;
	            $this->_tree( $xml_ref, $node->childNodes->item( 0 ), "" ) ;
	            $$xml_ref .= "</" . $node->tagName . ">\n" ;
        	} else {
	            $$xml_ref .= "$tab<" . join( ' ', @tag ) . ">\n" ;
	            
		        for ( my $n = 0 ; $n < $node->childNodes->length ; $n++ ) {
	              $this->_tree( $xml_ref, $node->childNodes->item( $n ), "$tab  " ) ;
        		}
	            
	            $$xml_ref .= "$tab</" . $node->tagName . ">\n" ;
        	}
        }
    }
}


#=======================================================================
# PHP

sub htmlspecialchars {
	my $this = shift ;
	my $text = shift ;
	
	$text =~ s/&/&amp;/g ;
	$text =~ s/</&lt;/g ;
	$text =~ s/>/&gt;/g ;
	$text =~ s/"/&quot;/g ;
	
	return $text ;
}


1;